---
typora-root-url: ./
header-includes: |
  <style>
  .reveal {
    font-size: 19pt;
  }
  .reveal ol,
  .reveal dl,
  .reveal ul {
    display: block;
    text-align: left;
    margin: 0 0 0 1em;
  }
  .reveal h1,
  .reveal h2,
  .reveal h3,
  .reveal h4,
  .reveal h5,
  .reveal h6 {
    margin: 0 0 20px 0;
    color: #222;
    font-weight: 300;
    line-height: 1.1;
    letter-spacing: normal;
    text-transform: none;
    text-shadow: none;
    word-wrap: break-word;
  }
  .reveal code {
    font-variant-ligatures: common-ligatures;
  }
  .reveal .progress {
    background: rgba(0, 0, 0, 0.2);
  }
  .reveal .progress span {
    background: #2a76dd;
    -webkit-transition: width 800ms cubic-bezier(0.26, 0.86, 0.44, 0.985);
    -moz-transition: width 800ms cubic-bezier(0.26, 0.86, 0.44, 0.985);
    transition: width 800ms cubic-bezier(0.26, 0.86, 0.44, 0.985);
  }
  .reveal .controls .navigate-left,
  .reveal .controls .navigate-left.enabled {
    border-right-color: #2a76dd; }

  .reveal .controls .navigate-right,
  .reveal .controls .navigate-right.enabled {
    border-left-color: #2a76dd; }

  .reveal .controls .navigate-up,
  .reveal .controls .navigate-up.enabled {
    border-bottom-color: #2a76dd; }
  .reveal .controls .navigate-down,
  .reveal .controls .navigate-down.enabled {
    border-top-color: #2a76dd; }
  .reveal .controls .navigate-left.enabled:hover {
    border-right-color: #6ca0e8; }
  .reveal .controls .navigate-right.enabled:hover {
    border-left-color: #6ca0e8; }
  .reveal .controls .navigate-up.enabled:hover {
    border-bottom-color: #6ca0e8; }
  .reveal .controls .navigate-down.enabled:hover {
    border-top-color: #6ca0e8; }
  </style>
author: ℤhichu ℂhen (ℝoss)
title: Futuristic Review of JACoW Editing
subtitle: A.K.A., how to lure innocent people with highfalutin jargons
date: 17 Dec, 2021
---

# Proposal Report on `JACoW` Evolution

## Disclaimer

- These slides speak only for the speaker, who has been an author for roughly twenty years and an editor for three years.
- Some opinions are taken from my friends. If you find some contents funny, it’s because they came from my friends.
- This is not an espionage to replace Human race with Cylons.

## Overview

- What do authors and editors want
- A new scheme & the implementation
- Futuristic future work
- Summary

## As an Author

- Nice web interface.
- Support for various formats.
- Intuitive visualisation between revisions.
- Versatile ways to download interested papers.

## As an Editor

- Standardise inputs from authors.
- Big screen with more contents and preferably bigger font sizes.
- Manoeuvre changes.
- Avoid duplicating jobs.
- Reference format is a curse.

## Efforts for Now

- Provide templates for authors and hope they’ll follow with crossed fingers.
- Lucky for `LaTeX`, we can use a decent editor with regular expression support.
- `Cat Scan Tools` by David Button.
- `https://ref.ipac19.org/` or directly go to the homepage for the reference if it’s a `JACoW` publication.
  - Sometimes, they do not give the same entry.
- Sharp eyes, steady hands, and QA.

# The Current Workflow of a Conference

- Install `SPMS`
- Registration
- Abstract submission
- Booklet
- Paper submission
- Editing/QA/Author-Title-Check…
- Proceedings



### Pre-Conference

<img src="jacow-workflow-pre.svg" alt="jacow-workflow-pre" style="width:100%;" />

### Editing

<img src="jacow-workflow-edit.svg" alt="jacow-workflow" style="width:100%;" />

Some people are colourblind. Maybe with another colour scheme?

## A New Design

Taking advantage of the migration of `JPSP`, and the software development during the past decades, I see an opportunity to design a different back-end:

- Papers uploaded by authors/editors are converted to Abstract Syntactic Trees (ASTs)
- ASTs are converted to various formats: `Word`, `OpenDocument`, `LaTeX`, `Markdown`…
- Each (or perhaps only the original one) upload is checked automatically for Author-Title-Check, syntactic errors, etc.
- Each change is monitored to be merged

### Back-end

<img src="ng-editing.svg" alt="ng-editing" style="width:100%;" />

### A Preliminary `pandoc` Filter Example

```haskell
{-# LANGUAGE OverloadedStrings #-}

import Data.Text (pack,unpack,Text)

import Text.Pandoc.Builder
import Text.Pandoc
import Text.Pandoc.Walk

import System.Environment (getArgs)

purely :: (b -> PandocPure a) -> b -> a
purely f = either (error . show) id . runPure . f

latex :: Text -> Pandoc
latex = purely $ readLaTeX def{
                   readerExtensions = getDefaultExtensions "latex" }

data Section = Section
  { level :: Int
  , attr  :: String
  , title :: String
  }

createSections :: Pandoc -> [Section]
createSections = query mkSec
  where mkSec (Header n _ inline) = [ Section
               { level=n
               ,  attr=""
               , title=getPlain inline
               }]
        mkSec _                   = []

getPlain :: [Inline] -> String
getPlain [] = ""
getPlain (x:xs) = stripeFrom x <> getPlain xs
  where stripeFrom (Str         strs  ) = unpack   strs
        stripeFrom (Code _      strs  ) = unpack   strs
        stripeFrom (Math _      strs  ) = unpack   strs
        stripeFrom (RawInline _ strs  ) = unpack   strs
        stripeFrom (Emph        strs  ) = getPlain strs
        stripeFrom (Underline   strs  ) = getPlain strs
        stripeFrom (Strong      strs  ) = getPlain strs
        stripeFrom (Strikeout   strs  ) = getPlain strs
        stripeFrom (Superscript strs  ) = getPlain strs
        stripeFrom (Subscript   strs  ) = getPlain strs
        stripeFrom (SmallCaps   strs  ) = getPlain strs
        stripeFrom (Quoted _    strs  ) = getPlain strs
        stripeFrom (Cite _      strs  ) = getPlain strs
        stripeFrom (Span _      strs  ) = getPlain strs
        stripeFrom (Link _      strs _) = getPlain strs
        stripeFrom (Image _     strs _) = getPlain strs
        stripeFrom Space                = " "
        stripeFrom _                    = ""


main :: IO ()
main = do
    (fileName:_) <- getArgs
    src <- readFile fileName
    let secs = createSections $ latex $ pack src
    putStrLn $ printResult $ getContents $ secs
    where getLevel (Section n _ str) = n
          getLevels = map getLevel
          getTitle (Section n attr str) = attr <> (show n) <> resetColor <> " " <> str
          diffl1 xs = zipWith (-) xs (0:xs)
          getColor n = if n>1 then boldRed else boldGreen
          boldRed    = "\x1b[1;31m"
          boldGreen  = "\x1b[1;32m"
          resetColor = "\x1b[0m"
          convertPretty (k, (Section n _ str)) = Section n (getColor k) str
          getContents sec = map convertPretty $ zip (diffl1 $ getLevels $ sec) sec
          printResult = unlines . map getTitle
```

#### Results

<img src="catscan-ng.png" alt="catscan-ng" style="width:100%;" />

## Future Works (Phase II)

### Patch Theory

<img src="ng-patching.svg" alt="ng-patching" style="width:100%;" />

- `darcs` already takes care of that. I don’t want to go too deep. If interested, please refer to
  - [`http://darcs.net/Theory/Example`](http://darcs.net/Theory/Example)

### Future Online Interface

<img src="ng-gui.svg" alt="ng-gui" style="width:100%;" />

### Future Online Publishing

In addition to the good old `pdf` files, an `HTML` output is also nice.

- The `HTML` is ready as soon as the `.md` is available. No extra work needed.

- Hyperlinks, or fancy interactive graphics with javascript.

- Meta goodies: times been cited, view similar papers by keywords/categories, follow-up works, share in Tweeter…

- With the plugin feature of `pandoc`, we can use code fences to post-process the code to generate outputs/figures online. Especially useful for Wikis.

  <img src="ng-plugin.svg" alt="ng-plugin" style="width:100%;" />

## Summary

- 
